// Variables used by Scriptable.
// These must be at the very top of the file. Do not edit.
// icon-color: deep-gray; icon-glyph: chart-line;

// MIT License

// Copyright (c) 2020 PQINA
// https://pqina.nl/

// Permission is hereby granted, free of charge, to any person obtaining 
// a copy of this software and associated documentation files (the "Software"), 
// to deal in the Software without restriction, including without limitation the 
// rights to use, copy, modify, merge, publish, distribute, sublicense, and/or 
// sell copies of the Software, and to permit persons to whom the Software
// is furnished to do so, subject to the following conditions:

// The above copyright notice and this permission notice shall
// be included in all copies or substantial portions of the Software.

// THE SOFTWARE IS PROVIDED "AS IS", WITHOUT WARRANTY OF ANY KIND, 
// EXPRESS OR IMPLIED, INCLUDING BUT NOT LIMITED TO THE WARRANTIES OF 
// MERCHANTABILITY, FITNESS FOR A PARTICULAR PURPOSE AND NONINFRINGEMENT. 
// IN NO EVENT SHALL THE AUTHORS OR COPYRIGHT HOLDERS BE LIABLE FOR ANY
// CLAIM, DAMAGES OR OTHER LIABILITY, WHETHER IN AN ACTION OF CONTRACT,
// TORT OR OTHERWISE, ARISING FROM, OUT OF OR IN CONNECTION WITH THE 
// SOFTWARE OR THE USE OR OTHER DEALINGS IN THE SOFTWARE.

// Gumroad bright theme
let chartDayLabelColor = '#999999'
let chartLineColor = '#459EA3'
let chartBarColor = '#EBEBEB'
let chartRecurringBarColor = '#cccccc'
let incomeLabelColor = '#6D6D69'
let lastUpdateLabelColor = '#BFBFBF'
let widgetBackgroundColor = '#FFFFFF'
let widgetBackgroundGradient = new LinearGradient()
widgetBackgroundGradient.locations = [0, 1]
widgetBackgroundGradient.colors = [new Color('#FFFFFF'), new Color('#FFFFFF')]

const autoAppearance = true

if (Device.isUsingDarkAppearance() && autoAppearance) {
  chartDayLabelColor = '#aaaaaa'
  chartLineColor = '#79C8C9'
  chartBarColor = '#5E5B5C'
  chartRecurringBarColor = '#4C4949'
  incomeLabelColor = '#EBEBEB'
  lastUpdateLabelColor = '#837D7D'
  widgetBackgroundColor = '#2F2D2E'
  widgetBackgroundGradient.colors = [new Color('#333030'), new Color('#393636')]
}

// layout
const chartLineDotRadius = 5
const chartLineWidth = 4
const chartBarMarginHorizontal = 6
const currency = 'USD'
const locale = 'en-US'

// gumroad access token
const token = args.widgetParameter

// exit if no token received
if (!token) return

// current date
const now = new Date()

// get all sale items
const sales = await fetchGumroadSales()

// no sales received, so exit
if (!sales) return

// calculate total income for period
const incomeTotal = Object.keys(sales).reduce((prev, day) => prev + sales[day].income, 0) / 100

const salesTotal = Object.keys(sales).reduce((prev, day) => prev + sales[day].sales, 0)

//
// widget
//
const widget = new ListWidget()
widget.setPadding(20, 0, 10, 0)
widget.backgroundColor = new Color(widgetBackgroundColor)
widget.backgroundGradient = widgetBackgroundGradient

// total income label
const incomePeriodTotal = widget.addText(getAsCurrency(incomeTotal))
incomePeriodTotal.textColor = new Color(incomeLabelColor)
incomePeriodTotal.font = new Font('Avenir-Medium', 18)
incomePeriodTotal.centerAlignText()

widget.addSpacer(null)


// chart
const chart = widget.addImage(drawSalesChart(sales))
chart.centerAlignImage()

widget.addSpacer(null)

// footer
const footerStack = widget.addStack()
footerStack.layoutHorizontally()
footerStack.spacing = 5
footerStack.centerAlignContent()
footerStack.setPadding(0, 14, 0, 12)


// total sales
const salesTotalIcon = addSymbolImage(footerStack, 'cube.box', 10, lastUpdateLabelColor)

const salesTotalText = footerStack.addText(`${salesTotal}`)
salesTotalText.font = new Font('AvenirNext-Medium', 10)
salesTotalText.textColor = new Color(lastUpdateLabelColor)



// push right
footerStack.addSpacer(null)


const lastUpdateIcon = addSymbolImage(footerStack, 'arrow.2.circlepath', 10, lastUpdateLabelColor)

const lastUpdateText = footerStack.addText(now.toLocaleTimeString([], { hour: '2-digit', minute: '2-digit' }))
lastUpdateText.font = new Font('AvenirNext-Medium', 10)
lastUpdateText.textColor = new Color(lastUpdateLabelColor)




// done!
config.runsInWidget ? Script.setWidget(widget) : widget.presentSmall()
Script.complete()


//
// chart draw methods
//
function drawSalesChart(sales) {

    const ctx = new DrawContext()
    ctx.size = new Size(300, 150)
    ctx.opaque = false

    drawBars(ctx, sales)
    drawLine(ctx, sales)
    drawDays(ctx, sales)

    return ctx.getImage()
}




function drawDays(ctx, sales) {
  
  const stepX = ctx.size.width / 7
  const height = 16
  const y = ctx.size.height - height
  const width = stepX
  
  ctx.setTextAlignedCenter()
  ctx.setTextColor(new Color(lastUpdateLabelColor))
  
  const margin = chartBarMarginHorizontal
  
  Object.keys(sales).forEach((date, i) => {
    const label = new Date(date).toLocaleDateString(locale, { weekday: 'short' })
    const x = i * stepX + margin * .5
    ctx.drawTextInRect(label.substr(0,2), new Rect(
      x,y, width, height
    ))
  })
}

function drawLine(ctx, sales) {

  // used for dots
  ctx.setFillColor(new Color(chartLineColor))

  const path = new Path()

  const incomeMax = getDayMaxIncome(sales)
  const r = chartLineDotRadius
  const height = ctx.size.height - r * 2 - 22
  const stepY = height / incomeMax
  const stepX = ctx.size.width / 7

  Object.keys(sales).forEach((date, i) => {
    
    const count = sales[date].income
    const x = (i * stepX) + (stepX / 2) + (chartBarMarginHorizontal *.5)
    const y = r + height - (count * stepY)
    const point = new Point(x, y)

    if (i===0) {
      path.move(point)
    } 
    else {
      path.addLine(point)
    }

    // skip dot if no sales
    if (sales[date].income <= 0) return

    // draw dot
    const dot = new Rect(x - r, y - r, r * 2, r * 2)
    ctx.fillEllipse(dot)
  })

  // draw line
  ctx.addPath(path)
  ctx.setLineWidth(chartLineWidth)
  ctx.setStrokeColor(new Color(chartLineColor))
  ctx.strokePath()
}

function drawBars(ctx, sales) {
  
  const salesMax = getDayMaxSales(sales)
  
  const width = ctx.size.width / 7
  const height = ctx.size.height - 56
  const step = height / salesMax
  const margin = chartBarMarginHorizontal
  
  Object.keys(sales).forEach((date, i) => {
    
    const count = sales[date].sales
    const recurring = sales[date].recurring
    
    const o = margin * .5
    const x = (i * width) + o
    const y = ctx.size.height - 22
    const w = width - margin
    let h = count * step
    
    ctx.setFillColor(new Color(chartBarColor))
    let bar = new Rect(x + o, y, w, -h)

    ctx.fillRect(bar)
    
    if (!recurring) return
    
    h = recurring * step
    
    ctx.setFillColor(new Color(chartRecurringBarColor))
    bar = new Rect(x + o, y, w, -h)
    ctx.fillRect(bar)
    
  })
}


//
// helper methods
//
function getDateOffsetFromNow(days) {
  let date = new Date(now)
  date.setDate(date.getDate() - days)
  date = flattenDate(date)
  return date
}

function toISODate(d) {
  const date = d.getDate()
  const month = d.getMonth() + 1
  const year = d.getFullYear()
  return `${year}-${pad(month)}-${pad(date)}`
}

function flattenDate(date) {
  date.setHours(0)
  date.setMinutes(0)
  date.setSeconds(0)
  date.setMilliseconds(0)
  return date
}

function getDateSlots() {
  const slots = {}
  for(let i = 6; i >= 0; i--) {
    const base = { income: 0, sales: 0, recurring: 0}
    slots[getDateOffsetFromNow(i)] = base
  }
  return slots
}

function getSalesPerDay(sales) {
  const slots = getDateSlots()
  sales
    .sort(sortSalesByDate)
    .forEach(sale => {
      
    const slot = flattenDate(new Date(sale.created_at))
    
    if (!slots[slot]) return
    
    slots[slot].sales += 1
    slots[slot].recurring += sale.recurring_charge ? 1 : 0
    slots[slot].income += sale.price
  })

  return slots
}

async function fetchGumroadSales() {
  const date = getDateOffsetFromNow(7)
  const url = `/v2/sales?after=${toISODate(date)}&page=1`
  const sales = await fetchGumroadPagedSales(url)
  return getSalesPerDay(sales.filter(sale => !sale.refunded))
}

async function fetchGumroadPagedSales(url) {
  const data = await new Request(`https://api.gumroad.com${url}&access_token=${token}`).loadJSON()
  if (data.next_page_url) {
    const sales = await fetchGumroadPagedSales(data.next_page_url)
    return data.sales.concat(sales)
  }
  return data.sales
}

function getDayMaxIncome(sales) {
  return Object.keys(sales).reduce(
    (prev, day) => {
      if (sales[day].income > prev) {
        prev = sales[day].income
      }
      return prev
    }, 
    0
  )
}

function getDayMaxSales(sales) {
  return Object.keys(sales).reduce(
    (prev, day) => {
      if (sales[day].sales > prev) {
        prev = sales[day].sales
      }
      return prev
    }, 
    0
  )
}

function pad(value) {
  return `${value}`.padStart(2, '0')
}

function sortSalesByDate(a, b) {
    const aDate = new Date(a.created_at)
    const bDate = new Date(b.created_at)
    if (aDate < bDate) return -1
    if (aDate > bDate) return 1
    return 0
}

function addSymbolImage(ctx, name, size, color) {
  const symbol = SFSymbol.named(name)
  symbol.applyFont(Font.systemFont(size))
  const image = symbol.image
  const ctxImage = ctx.addImage(image)
  ctxImage.imageSize = new Size(size, size)
  ctxImage.tintColor = new Color(color)
  return ctxImage
}

function getAsCurrency(value) {
  const formatter = new Intl.NumberFormat(locale, { style: 'currency', currency: currency, })
  return formatter.format(value)
} 
